# Script to build a SIG installation package:
# - Ensures necessary COTS binaries are in place.
# - Updates SIG version in sqwm.xml.
# - Copies only the desired 'env' directories.

# Fail fast on any errors
$ErrorActionPreference = "Stop"

# TODO: what if we capture more than one tag here?
# Just use 'unknown' if we can't detect the hg attributes.
try
{
   $sigTag = hg parents --template "{tags}"
   if ([String]$sigTag -eq '')
   {
      $sigTag = 'untagged'
   }
   $sigChangeset = hg parents --template "{node|short}"
   $hgStatus = @(hg status)
   if ($hgStatus.Count -gt 0)
   {
      Write-Warning "Not a clean working directory!"
      Write-Output $hgStatus
      $sigTag = 'unknown'
      $sigChangeset += '+'
   }
}
catch [System.Management.Automation.CommandNotFoundException]
{
    Write-Warning "Could not detect version! Using 'unknown'."
	$sigTag = 'unknown'
	$sigChangeset = 'unknown'
}
$sourceDir = 'install'
$targetDir = "target\$sigTag"

Write-Output "Creating installer for SIG $sigTag."

if (Test-Path $targetDir -PathType Container)
{
   "Directory '$targetDir' already exists but too careful to delete it. Aborting."
   return
}

Write-Output "Creating target directory '$targetDir'..."
mkdir $targetDir | Out-Null
# This will copy the 'env' directory but not its contents.
Copy-Item $sourceDir\* $targetDir\ 

# Copy desired 'env' directories.
Write-Output "Copying 'env' directories..."
Copy-Item -recurse $sourceDir\env\dev $targetDir\env\ 
Copy-Item -recurse $sourceDir\env\sqa $targetDir\env\ 
Copy-Item -recurse $sourceDir\env\pre $targetDir\env\ 
Copy-Item -recurse $sourceDir\env\prod $targetDir\env\ 
Copy-Item -recurse $sourceDir\env\common $targetDir\env\ 

# Update sqwm.xml with the SIG version.
Write-Output "Updating sqwm.xml with SIG version and alert emails..."
$sqwmXmlFile = "$targetDir\env\common\sqwm.xml"
$sigVersion = "$sigTag (changeset $sigChangeset)"
$commonSqwmXml = (Get-Content "$sqwmXmlFile") |
   %{
      $_ -replace 'SIG version: unknown', "SIG version: $sigVersion" -replace
         'smtp.xx.ext    ', 'smtp.xx.ext' }
# Replace emails per-environment. Note that I originally wanted to do this with
# XSLT but that made some miscellaneous transformations to the output (e.g.,
# replacing &apos; with an actual apostrophe) and caused Mirth Connect to reject
# the XML on import. So we just kludge it with a string replacement.
$emailsOrig = '<string>user12@domain.ext</string>'
$emailsDev  = '<string>user12@domain.ext</string><string>first.last12@xx.ext</string><string>user.useruser@xx.ext</string>'
# TODO: remove developer emails after IOC
$emailsProd = '<string>SQWM.Alerts@ge.com</string><string>user12@domain.ext</string><string>first.last12@xx.ext</string><string>user.useruser@xx.ext</string>'
$emailTable = @{
   dev = $emailsDev;
   sqa = $emailsDev;
   pre = $emailsDev;
   prod = $emailsProd
}
foreach ($pair in $emailTable.GetEnumerator())
{
   $envName = $pair.key
   $emails = $pair.value
   $targetPath = "$targetDir\env\$envName\sqwm.xml"
   Write-Output "Writing $targetPath..."
   $commonSqwmXml |
      %{$_ -replace $emailsOrig, $emails } |
      Out-File -Encoding ASCII -filepath $targetPath
}
# Remove now-useless common sqwm.xml
rm $sqwmXmlFile

# Update sqwm.properties with the SIG version.
Write-Output "Updating sqwm.properties with SIG version..."
foreach ($f in dir -recurse "$targetDir\env" | where { $_.Name -ieq 'sqwm.properties' })
{
   $fullName = $f.FullName
   "Updating $fullName..."
   # Get-Content takes a String so make sure we give it the full path.
   (Get-Content $fullName) |
      %{$_ -replace 'configVersion = .*', "configVersion = $sigVersion"} |
      Out-File -Encoding ASCII -filepath $fullName
}

# Copy 'tools' directory's contents
Write-Output "Copying tools..."
Copy-Item -recurse $sourceDir\tools\* $targetDir\tools\

# Copy 'bin' directory's contents if it exists.
Write-Output "Copying binary installers..."
if (Test-Path $sourceDir\bin -PathType Container)
{
   Copy-Item -recurse $sourceDir\bin\* $targetDir\bin\
}

# Ensure correct COTS installers are in place.
Write-Output "Checking installer..."
Push-Location $targetDir
try
{
   .\install.cmd check
   if (!$?) { throw "Installation package check failed!" }
}
finally # Pop-Location even if we threw an Exception
{
   Pop-Location
}

Write-Output "Done."

